﻿using System;
using System.ComponentModel;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;

using Framework.Data;

namespace Framework.Web.UI
{
    /// <summary>
    /// マッピング機能を持ったドロップダウンリストです。
    /// </summary>
    [DefaultProperty("MappingName")]
    [ToolboxData("<{0}:MDropDownList runat=server />")]
    public class MDropDownList : DropDownList, IMappingControl, IInputControl, ILocalizeControl, IListControl
    {

        /// <summary>
        /// マッピング名
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string MappingName {get; set;}

        /// <summary>
        /// 複数項目で主キーになる場合は、このプロパティを設定し、MappingNameに複数項目をここで設定した値をつないでセットします。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public char MappingNameSeparator { get; set; }

        /// <summary>
        /// 初期値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string DefaultValue { get; set; }

        /// <summary>
        /// リストキー
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ListKey {get; set;}

        /// <summary>
        /// リストソースのリクエストイベント
        /// </summary>
        [Category("Custom")]
        public event ListSrcRequestEventHandler ListSrcRequest;

        /// <summary>
        /// 必須かどうか。選択項目のValue値がBlankValueに等しいときに未入力とみなす。
        /// </summary>
        [Category("Custom")]
        public bool IsRequired
        {
            get { return _isRequired; }
            set
            {
                _isRequired = value;

                //必須の場合は空行が必要
                if (_isRequired)
                {
                    this.WithBlank = true;
                }
            }
        }
        private bool _isRequired = false;

        /// <summary>
        /// 空行を追加するかどうか
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool WithBlank { get; set; }

        /// <summary>
        /// 空行を追加したときの、空行のValue値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string BlankValue { get; set; }

        /// <summary>
        /// 空行を追加したときの、空行のText値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string BlankText { get; set; }

        /// <summary>
        /// 依存コントロールのID。ここで指定したコントロールの選択状態が変わった場合は、自身のアイテムを変更する。
        /// 現在指定できるのはMDropDownListだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ControlToDepend { get; set; }
        private MDropDownList _dependDDL;

        /// <summary>
        /// 依存コントロールのID。ここで指定したコントロールの選択状態が変わった場合は、自身のアイテムを変更する。
        /// 現在指定できるのはLGridViewだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string GridViewsToDepend { get; set; }
        private LGridView[] _dependGVWs;

        /// <summary>
        /// 依存コントロールのID。
        /// 現在指定できるのはMTextBoxだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string TextBoxesToDepend { get; set; }
        private MTextBox[] _dependTXTs;

        /// <summary>
        /// アイテムが選択された場合に、入力が必須になるコントロールを指定します。
        /// </summary>
        [Category("Custom")]
        public string ControlsToRequired { get; set; }

        /// <summary>
        /// リストアイテムのツールチップを表示するかどうか。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool EnableToolTip { get; set; }

        /// <summary>
        /// テキスト値をキーにして、テキストをローカライズするかどうか？
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool LocalizeByText { get; set; }

        /// <summary>
        /// アイテムがローカライズされているときはtrueになります。
        /// </summary>
        private bool _localized = false;
        
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MDropDownList()
        {
            this.EnableToolTip = false;
            this.LocalizeByText = false;
        }

        /// <summary>
        /// OnInit
        /// </summary>
        /// <param name="e"></param>
        protected override void OnInit(EventArgs e)
        {
            //依存コントロールがある場合は、依存コントロールのChangeイベントを受け取る。
            if (this.ControlToDepend.IsNotEmpty())
            {
               _dependDDL = this.NamingContainer.FindControl(this.ControlToDepend) as MDropDownList;
               if (_dependDDL != null)
                {
                    _dependDDL.AutoPostBack = true;
                    _dependDDL.SelectedIndexChanged += new EventHandler(dependDDL_SelectedIndexChanged);
                }
            }
            if (this.GridViewsToDepend.IsNotEmpty())
            {
                _dependGVWs = this.GridViewsToDepend.SplitByConma()
                    .Select(t => this.NamingContainer.FindControl(t.Trim()) as LGridView)
                    .ToArray();
                _dependGVWs.Foreach(gvw =>
                {
                    gvw.SelectedIndexChanged += new EventHandler(dependDDL_SelectedIndexChanged);
                });
            }

            base.OnInit(e);
        }

        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            if (this.EnableToolTip == true)
            {
                foreach (ListItem item in base.Items)
                {
                    item.Attributes["title"] = item.Text;
                }

                this.Attributes["onmouseover"] = "this.title=this.options[this.selectedIndex].title";
            }

            base.OnPreRender(e);
        }

        /// <summary>
        /// 依存しているDDLの選択が変更されたときに呼び出されるイベントです。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void dependDDL_SelectedIndexChanged(object sender, EventArgs e)
        {
            //リストソースを取得し、データバインド
            var arg = (string)null;
            if (sender is ListControl)
            {
                arg = ((ListControl)sender).SelectedValue;
            }
            else if (sender is LGridView)
            {
                arg = ((LGridView)sender).SelectedDataKeyValues;
            }
            this.ListSrcDataBind(arg);

            //自分が依存コントロールに指定されている可能性があるので、変更を通知する。
            this.OnSelectedIndexChanged(new EventArgs());
        }

        /// <summary>
        /// OnDataBound
        /// </summary>
        /// <param name="e"></param>
        protected override void OnDataBound(EventArgs e)
        {
            if (this.WithBlank)
            {
                //空行を追加
                base.Items.Insert(0, new ListItem(this.BlankText, this.BlankValue));
            }

            base.OnDataBound(e);
        }

        #region IMappingControl メンバ

        /// <summary>
        /// dataに選択項目の値を追加する。MappingNameSeparatorが指定してある場合は複数項目がセットされます。
        /// </summary>
        /// <param name="data"></param>
        public void RequestMappingData(Framework.Data.MappingData data)
        {
            if (this.MappingNameSeparator.IsNotMinValue())
            {
                var mappingNames = this.MappingName.Split(this.MappingNameSeparator);
                var values = base.SelectedValue.Split(this.MappingNameSeparator);
                for (int i = 0; i < mappingNames.Length && i < values.Length; i++)
                {
                    data.Add(mappingNames[i], values[i]);
                }
            }
            else
            {
                data.Add(this.MappingName, base.SelectedValue);
            }
        }

        /// <summary>
        /// リスト値をセットして、その後dataに応じて項目の選択状態を変えます。
        /// </summary>
        /// <param name="data"></param>
        virtual public void SetMappingData(Framework.Data.MappingData data)
        {
            base.ClearSelection();

            //依存するコントロールがある場合は、それをさきにマッピングする。
            string arg = null;
            if (_dependDDL != null)
            {
                _dependDDL.SetMappingData(data);
                arg = _dependDDL.SelectedValue;
            }
            if (_dependGVWs != null)
            {
                _dependGVWs.Foreach(gvw =>
                {
                    gvw.SetMappingData(data);
                    arg = gvw.SelectedDataKeyValues;
                });
            }
            if (this.TextBoxesToDepend.IsNotEmpty())
            {
                this.TextBoxesToDepend.SplitByConma().Foreach(id =>
                {
                    var txt = this.FindControl(id.Trim()) as MTextBox;
                    txt.SetMappingData(data);
                    arg = txt.Text;
                });
            }

            //リストソースを取得し、データバインド
            this.ListSrcDataBind(arg);

            //マッピング
            var val = (string)null;
            if (this.MappingName.IsNotEmpty())
            {
                if (this.MappingNameSeparator.IsNotMinValue())
                {
                    var mappingNames = this.MappingName.Split(this.MappingNameSeparator);
                    val = string.Join(",", mappingNames.Select(name => (data[name] == null) ? string.Empty : data[name]).ToArray());
                }
                else
                {
                    val = data[this.MappingName];
                }
            }

            if (val != null)
            {
                if (base.Items.FindByValue(val) != null)
                {
                    base.SelectedValue = val;
                }
            }
            else if (this.DefaultValue != null)
            {
                if (base.Items.FindByValue(this.DefaultValue) != null)
                {
                    base.SelectedValue = this.DefaultValue;
                }
            }
        }

        /// <summary>
        /// リストソースをリクエストし、その戻りのデータをバインドする。
        /// </summary>
        public void ListSrcDataBind()
        {
            ListSrcDataBind(null);
        }

        /// <summary>
        /// リストソースをリクエストし、その戻りのデータをバインドする。
        /// </summary>
        /// <param name="arg"></param>
        public void ListSrcDataBind(string arg)
        {
            //リストソースを取得し、データバインド
            if (this.ListKey.IsNotEmpty())
            {
                var listSrc = (ListSrcDataSet)null;
                if (this.ListSrcRequest != null)
                {
                    var e = new ListSrcRequestEventArgs()
                    {
                        ListKey = this.ListKey,
                        Argument = arg
                    };
                    this.ListSrcRequest(this, e);
                    listSrc = e.ListSrc;
                }
                else
                {
                    listSrc = ListMgr.GetListSrc(this.ListKey);
                }

                //テキストのローカライズ
                if (this.LocalizeByText == true)
                {
                    listSrc.ListSrc.Foreach(row =>
                    {
                        row.Text = LTextMgr.GetText(row.Text);
                    });

                    _localized = true;
                }

                //Value値の異なるリストでバインドし直した時にエラーになったので、
                //その対処としてこの処理を入れる。なぜこれでよいかは不明です。（2010.3.5 suda）
                base.DataBind();

                base.DataValueField = "Value";
                base.DataTextField = "Text";
                base.DataSource = listSrc;
                base.DataBind();
            }
        }

        #endregion

        #region IInputControl メンバ

        public bool IsValid
        {
            get { return _isValid; }
            set { _isValid = value; }
        }
        private bool _isValid = true;

        public System.Web.UI.Control GetErrorControl()
        {
            return this;
        }

        /// <summary>
        /// 入力された場合にTrueを返します。DefaultValueが指定してある場合は選択をDefaultValue以外にした場合、
        /// BlankValueが指定してある場合は選択をBlankValue以外にした場合にTrueを返します。その他の場合は常にtrueを返します。
        /// </summary>
        /// <returns></returns>
        public bool IsInputed()
        {
            if (this.DefaultValue.IsNotEmpty())
            {
                return (this.DefaultValue != base.SelectedValue);
            }

            if (this.BlankValue.IsNotEmpty())
            {
                return (this.BlankValue != base.SelectedValue);
            }

            return true;
        }

        /// <summary>
        /// 選択された項目の値とBlankValueが一致している場合にTrueを返します。
        /// </summary>
        /// <returns></returns>
        public bool IsBlank()
        {
            return (base.SelectedValue == this.BlankValue);
        }

        /// <summary>
        /// 入力値の検証
        /// </summary>
        /// <returns></returns>
        public bool ValidateInput(bool dataTypeOnly)
        {
            if (dataTypeOnly == true) return true;

            //必須チェック
            if (this.IsRequired && this.IsBlank())
            {
                return false;
            }

            //要求コントロールの必須チェック
            if (this.ControlsToRequired.IsNotEmpty() && this.IsBlank() == false)
            {
                bool bRet = true;
                foreach (var s in this.ControlsToRequired.SplitByConma().Select(s=>s.Trim()))
                {
                    var id = s.Replace("{0}", base.SelectedValue);
                    var ic = base.NamingContainer.FindControl(id) as IInputControl;
                    if (ic.IsBlank())
                    {
                        bRet = false;
                        ic.SetValidateResult(false);
                    }
                    else
                    {
                        ic.SetValidateResult(true);
                    }
                }

                return bRet;
            }

            return true;
        }

        public void SetValidateResult(bool isSuccess)
        {
            this.IsValid = (this.IsValid && isSuccess);
            base.BackColor = this.IsValid ? UIController.ValidateSuccessColor : UIController.ValidateErrorColor;
        }

        public void ClearInput()
        {
            if (this.DefaultValue != null)
            {
                if (base.Items.FindByValue(this.DefaultValue) != null)
                {
                    base.SelectedValue = this.DefaultValue;
                }
            }
            else
            {
                base.ClearSelection();
            }
        }

        public string GetInputValue()
        {
            return base.SelectedValue;
        }

        #endregion

        #region ILocalizeControl メンバ

        public void Localize(System.Globalization.CultureInfo lang)
        {
            if (this.LocalizeByText == true && _localized == false)
            {
                foreach (ListItem item in base.Items)
                {
                    item.Text = LTextMgr.GetText(lang, item.Text);
                }

                _localized = true;
            }
        }

        #endregion
    }
}
